<?php
// Run in background: no browser output
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once '../config/database.php';

if (php_sapi_name() !== 'cli') {
    die('CLI only');
}

$batch_id = isset($argv[1]) ? intval($argv[1]) : 0;

if ($batch_id > 0) {
    processEmailImport($batch_id);
}

function processEmailImport($batch_id) {
    $db = getDB();
    
    // Get batch info
    $stmt = $db->prepare("
        SELECT ib.*, u.email as importer_email 
        FROM email_import_batches ib 
        JOIN users u ON ib.imported_by = u.id 
        WHERE ib.id = ?
    ");
    $stmt->bind_param("i", $batch_id);
    $stmt->execute();
    $batch = $stmt->get_result()->fetch_assoc();
    
    if (!$batch) {
        return;
    }
    
    $filepath = '../uploads/email_lists/' . $batch['filename'];
    
    if (!file_exists($filepath)) {
        updateBatchStatus($batch_id, 'failed');
        return;
    }
    
    $imported = 0;
    $failed = 0;
    $total = 0;
    
    // Process CSV file
    if (($handle = fopen($filepath, "r")) !== FALSE) {
        while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
            $total++;
            
            // Skip header if exists
            if ($total == 1 && (strpos($data[0], 'email') !== false)) {
                continue;
            }
            
            if (count($data) >= 2) {
                $email = trim($data[0]);
                $password = trim($data[1]);
                
                if (filter_var($email, FILTER_VALIDATE_EMAIL) && !empty($password)) {
                    // Check if email exists
                    $check = $db->prepare("SELECT id FROM email_pool WHERE email_address = ?");
                    $check->bind_param("s", $email);
                    $check->execute();
                    
                    if ($check->get_result()->num_rows == 0) {
                        // Insert new email
                        $stmt = $db->prepare("INSERT INTO email_pool (email_address, email_password) VALUES (?, ?)");
                        $stmt->bind_param("ss", $email, $password);
                        
                        if ($stmt->execute()) {
                            $imported++;
                        } else {
                            $failed++;
                        }
                    } else {
                        $failed++; // Duplicate
                    }
                } else {
                    $failed++;
                }
            }
            
            // Update progress every 100 records
            if ($total % 100 == 0) {
                updateImportProgress($batch_id, $imported, $failed);
            }
        }
        fclose($handle);
        
        // Final update
        updateImportProgress($batch_id, $imported, $failed);
        updateBatchStatus($batch_id, 'completed');
        
        // Send notification
        sendImportNotification($batch['importer_email'], $batch['batch_name'], $imported, $failed);
    }
}

function updateImportProgress($batch_id, $imported, $failed) {
    $db = getDB();
    $stmt = $db->prepare("
        UPDATE email_import_batches 
        SET total_emails = ?, imported_count = ?, failed_count = ? 
        WHERE id = ?
    ");
    $total = $imported + $failed;
    $stmt->bind_param("iiii", $total, $imported, $failed, $batch_id);
    $stmt->execute();
}

function updateBatchStatus($batch_id, $status) {
    $db = getDB();
    $stmt = $db->prepare("UPDATE email_import_batches SET status = ? WHERE id = ?");
    $stmt->bind_param("si", $status, $batch_id);
    $stmt->execute();
}

function sendImportNotification($email, $batch_name, $imported, $failed) {
    $subject = "Email Import Completed: " . $batch_name;
    $message = "Import completed!\n\n";
    $message .= "Batch: " . $batch_name . "\n";
    $message .= "Imported: " . $imported . " emails\n";
    $message .= "Failed: " . $failed . " emails\n\n";
    $message .= "Date: " . date('Y-m-d H:i:s');
    
    // In production, use PHPMailer
    mail($email, $subject, $message);
}