<?php
require_once 'config/database.php';

function assignEmailsToUser($user_id, $service_id, $quantity) {
    $db = getDB();
    
    // Get available emails from pool
    $stmt = $db->prepare("SELECT id FROM email_pool WHERE status = 'available' LIMIT ?");
    $stmt->bind_param("i", $quantity);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $assigned_emails = [];
    while ($row = $result->fetch_assoc()) {
        $assigned_emails[] = $row['id'];
    }
    
    if (count($assigned_emails) < $quantity) {
        return ['success' => false, 'message' => 'Not enough emails available'];
    }
    
    // Assign each email
    foreach ($assigned_emails as $email_id) {
        // Update email status
        $stmt = $db->prepare("UPDATE email_pool SET status = 'assigned' WHERE id = ?");
        $stmt->bind_param("i", $email_id);
        $stmt->execute();
        
        // Create purchase record
        $stmt = $db->prepare("INSERT INTO user_purchases (user_id, service_id, email_pool_id, quantity, used_count) VALUES (?, ?, ?, 1, 0)");
        $stmt->bind_param("iii", $user_id, $service_id, $email_id);
        $stmt->execute();
    }
    
    return ['success' => true, 'assigned' => count($assigned_emails)];
}

function useEmailForService($user_id, $email_id, $service_name) {
    $db = getDB();
    
    // Start transaction
    $db->begin_transaction();
    
    try {
        // Check if user owns this email for this service
        $stmt = $db->prepare("
            SELECT up.id, up.used_count, up.quantity 
            FROM user_purchases up 
            JOIN services s ON up.service_id = s.id 
            WHERE up.user_id = ? 
            AND up.email_pool_id = ? 
            AND s.service_name = ? 
            AND up.status = 'active'
            AND up.used_count < up.quantity
        ");
        $stmt->bind_param("iis", $user_id, $email_id, $service_name);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows == 0) {
            throw new Exception("Email not available for this service");
        }
        
        $purchase = $result->fetch_assoc();
        
        // Increment used count
        $stmt = $db->prepare("UPDATE user_purchases SET used_count = used_count + 1 WHERE id = ?");
        $stmt->bind_param("i", $purchase['id']);
        $stmt->execute();
        
        // If this was the last use, mark as used
        if (($purchase['used_count'] + 1) >= $purchase['quantity']) {
            $stmt = $db->prepare("UPDATE email_pool SET status = 'used' WHERE id = ?");
            $stmt->bind_param("i", $email_id);
            $stmt->execute();
            
            $stmt = $db->prepare("UPDATE user_purchases SET status = 'used' WHERE id = ?");
            $stmt->bind_param("i", $purchase['id']);
            $stmt->execute();
        }
        
        // Log usage
        $ip = $_SERVER['REMOTE_ADDR'];
        $stmt = $db->prepare("INSERT INTO email_usage_log (user_id, email_pool_id, service_id, action_type, ip_address) VALUES (?, ?, (SELECT id FROM services WHERE service_name = ?), 'used', ?)");
        $stmt->bind_param("iiss", $user_id, $email_id, $service_name, $ip);
        $stmt->execute();
        
        $db->commit();
        return ['success' => true];
        
    } catch (Exception $e) {
        $db->rollback();
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

function importEmailsToPool($emails) {
    $db = getDB();
    $imported = 0;
    $errors = [];
    
    foreach ($emails as $index => $email_data) {
        $email = $email_data['email'];
        $password = $email_data['password'];
        
        // Check if email already exists
        $stmt = $db->prepare("SELECT id FROM email_pool WHERE email_address = ?");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        
        if ($stmt->get_result()->num_rows == 0) {
            // Insert new email
            $stmt = $db->prepare("INSERT INTO email_pool (email_address, email_password) VALUES (?, ?)");
            $stmt->bind_param("ss", $email, $password);
            
            if ($stmt->execute()) {
                $imported++;
            } else {
                $errors[] = "Failed to import email at line " . ($index + 1);
            }
        }
    }
    
    return ['imported' => $imported, 'errors' => $errors];
}
?>