<?php
require_once '../config/database.php';

// Telegram Bot Configuration
define('BOT_TOKEN', 'YOUR_BOT_TOKEN_HERE');
define('WEBHOOK_URL', 'https://yourdomain.com/telegram/webhook.php');

// Handle webhook or CLI
if (php_sapi_name() === 'cli') {
    // Set webhook from command line
    if (isset($argv[1]) && $argv[1] === 'setwebhook') {
        setWebhook();
    } elseif (isset($argv[1]) && $argv[1] === 'deletewebhook') {
        deleteWebhook();
    } else {
        echo "Usage: php bot.php [setwebhook|deletewebhook]\n";
    }
    exit;
}

// Webhook handler
$content = file_get_contents("php://input");
$update = json_decode($content, true);

if ($update) {
    processUpdate($update);
}

function processUpdate($update) {
    $db = getDB();
    
    if (isset($update['message'])) {
        $message = $update['message'];
        $chat_id = $message['chat']['id'];
        $telegram_id = $message['from']['id'];
        $username = isset($message['from']['username']) ? $message['from']['username'] : '';
        $text = isset($message['text']) ? $message['text'] : '';
        
        // Save or update user
        $stmt = $db->prepare("
            INSERT INTO users (telegram_id, telegram_username, name) 
            VALUES (?, ?, ?)
            ON DUPLICATE KEY UPDATE 
            telegram_username = VALUES(telegram_username),
            name = VALUES(name)
        ");
        $name = isset($message['from']['first_name']) ? $message['from']['first_name'] : 'User';
        $stmt->bind_param("sss", $telegram_id, $username, $name);
        $stmt->execute();
        
        // Save message log
        $stmt = $db->prepare("
            INSERT INTO telegram_logs (telegram_id, chat_id, message_type, message, direction) 
            VALUES (?, ?, 'text', ?, 'incoming')
        ");
        $stmt->bind_param("sis", $telegram_id, $chat_id, $text);
        $stmt->execute();
        
        // Process command
        processCommand($telegram_id, $chat_id, $text, $username);
    }
    
    $db->close();
}

function processCommand($telegram_id, $chat_id, $text, $username) {
    $db = getDB();
    
    switch ($text) {
        case '/start':
            $message = "👋 Welcome *$username*!\n\n";
            $message .= "I'm your Email Service Bot.\n";
            $message .= "Available commands:\n";
            $message .= "/register - Register/Login to account\n";
            $message .= "/buy - Buy emails\n";
            $message .= "/myemails - View your emails\n";
            $message .= "/balance - Check balance\n";
            $message .= "/help - Show help\n";
            sendMessage($chat_id, $message);
            break;
            
        case '/register':
            // Check if already registered
            $stmt = $db->prepare("SELECT id, email FROM users WHERE telegram_id = ?");
            $stmt->bind_param("s", $telegram_id);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                $user = $result->fetch_assoc();
                if (!empty($user['email'])) {
                    sendMessage($chat_id, "✅ You are already registered!\nEmail: " . $user['email']);
                } else {
                    sendMessage($chat_id, "Please complete your registration on our website.");
                }
            } else {
                // Generate registration link
                $token = bin2hex(random_bytes(32));
                $registration_link = SITE_URL . "telegram-register.php?token=" . $token;
                
                // Save token
                $stmt = $db->prepare("
                    INSERT INTO telegram_sessions (telegram_id, chat_id, step, temp_data) 
                    VALUES (?, ?, 'registration', ?)
                    ON DUPLICATE KEY UPDATE 
                    step = VALUES(step),
                    temp_data = VALUES(temp_data)
                ");
                $stmt->bind_param("sis", $telegram_id, $chat_id, $token);
                $stmt->execute();
                
                $message = "🔗 *Registration Link*\n\n";
                $message .= "Click below to register:\n";
                $message .= "$registration_link\n\n";
                $message .= "This link is valid for 30 minutes.";
                sendMessage($chat_id, $message);
            }
            break;
            
        case '/myemails':
            sendUserEmails($telegram_id, $chat_id);
            break;
            
        case '/buy':
            sendServicesList($chat_id);
            break;
            
        default:
            // Check if it's a service selection
            if (strpos($text, 'buy_') === 0) {
                $service_id = str_replace('buy_', '', $text);
                processPurchase($telegram_id, $chat_id, $service_id);
            }
            break;
    }
}

function sendUserEmails($telegram_id, $chat_id) {
    $db = getDB();
    
    $stmt = $db->prepare("
        SELECT u.id, u.name 
        FROM users u 
        WHERE u.telegram_id = ?
    ");
    $stmt->bind_param("s", $telegram_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $user = $result->fetch_assoc();
        $user_id = $user['id'];
        
        // Get user's emails
        $stmt = $db->prepare("
            SELECT ep.email_address, s.service_name, ep.status 
            FROM email_pool ep
            JOIN user_purchases up ON ep.id = up.email_pool_id
            JOIN services s ON up.service_id = s.id
            WHERE up.user_id = ? AND up.status = 'active'
            ORDER BY ep.status, s.service_name
            LIMIT 50
        ");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $emails = $stmt->get_result();
        
        if ($emails->num_rows > 0) {
            $message = "📧 *Your Emails*\n\n";
            while ($email = $emails->fetch_assoc()) {
                $status_icon = $email['status'] == 'available' ? '✅' : '⚠️';
                $message .= "{$status_icon} *{$email['service_name']}*\n";
                $message .= "Email: `{$email['email_address']}`\n";
                $message .= "Status: {$email['status']}\n\n";
            }
            
            if ($emails->num_rows == 50) {
                $message .= "⚠️ Showing first 50 emails. View more on website.";
            }
        } else {
            $message = "📭 No emails found.\nUse /buy to purchase emails.";
        }
    } else {
        $message = "Please register first using /register";
    }
    
    sendMessage($chat_id, $message);
}

function sendServicesList($chat_id) {
    $db = getDB();
    $services = $db->query("SELECT id, service_name, price_per_email FROM services ORDER BY service_name");
    
    $message = "🛒 *Available Services*\n\n";
    
    while ($service = $services->fetch_assoc()) {
        $message .= "✨ *{$service['service_name']}*\n";
        $message .= "Price: \${$service['price_per_email']} per email\n";
        $message .= "Buy: `/buy_{$service['id']}`\n\n";
    }
    
    $message .= "Minimum purchase: 10 emails\n";
    $message .= "To buy, type: `/buy_[service_id]`";
    
    sendMessage($chat_id, $message);
}

function processPurchase($telegram_id, $chat_id, $service_id) {
    $db = getDB();
    
    // Get user
    $stmt = $db->prepare("SELECT id FROM users WHERE telegram_id = ?");
    $stmt->bind_param("s", $telegram_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $user = $result->fetch_assoc();
        $user_id = $user['id'];
        
        // Get service info
        $stmt = $db->prepare("SELECT service_name, price_per_email FROM services WHERE id = ?");
        $stmt->bind_param("i", $service_id);
        $stmt->execute();
        $service = $stmt->get_result()->fetch_assoc();
        
        if ($service) {
            // Save purchase intent
            $stmt = $db->prepare("
                INSERT INTO telegram_sessions (telegram_id, chat_id, step, temp_data) 
                VALUES (?, ?, 'purchase', ?)
                ON DUPLICATE KEY UPDATE 
                step = VALUES(step),
                temp_data = VALUES(temp_data)
            ");
            $data = json_encode(['service_id' => $service_id]);
            $stmt->bind_param("sis", $telegram_id, $chat_id, $data);
            $stmt->execute();
            
            $message = "🛒 *Purchase Request*\n\n";
            $message .= "Service: *{$service['service_name']}*\n";
            $message .= "Price: \${$service['price_per_email']} per email\n\n";
            $message .= "How many emails do you want to purchase?\n";
            $message .= "Reply with a number (10-1000)";
            
            sendMessage($chat_id, $message);
        } else {
            sendMessage($chat_id, "❌ Service not found.");
        }
    } else {
        sendMessage($chat_id, "Please register first using /register");
    }
}

function sendMessage($chat_id, $text) {
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendMessage";
    
    $data = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => 'Markdown',
        'disable_web_page_preview' => true
    ];
    
    $options = [
        'http' => [
            'method' => 'POST',
            'header' => "Content-Type: application/json\r\n",
            'content' => json_encode($data)
        ]
    ];
    
    $context = stream_context_create($options);
    file_get_contents($url, false, $context);
    
    // Log outgoing message
    $db = getDB();
    $stmt = $db->prepare("
        INSERT INTO telegram_logs (telegram_id, chat_id, message_type, message, direction) 
        VALUES (?, ?, 'text', ?, 'outgoing')
    ");
    $telegram_id = ''; // You need to track this
    $stmt->bind_param("sis", $telegram_id, $chat_id, $text);
    $stmt->execute();
}

function setWebhook() {
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/setWebhook?url=" . urlencode(WEBHOOK_URL);
    $result = file_get_contents($url);
    echo "Webhook set: " . $result . "\n";
}

function deleteWebhook() {
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/deleteWebhook";
    $result = file_get_contents($url);
    echo "Webhook deleted: " . $result . "\n";
}